/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *
 * Hashtable - a Hashtable that uses ints as the keys
 *
 * based on IntHashtable by the ACME Labs:
 * <a href="http://www.acme.com/java/software/Acme.IntHashtable.html">ACME IntHashtable</a>
 *
 * This one is highly optimized by Daniel Schregenberger to be used 
 * for the dynamic checks of the universe type system.
 *
 * It was ported to Scala by Manfred Stock.
 *
 * $Id: Hashtable.scala 883 2008-02-01 18:59:56Z ms $
 */

package ch.ethz.inf.sct.uts.rt.implementation

import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;

/**
 * Hashtable with integer keys.
 * 
 * @author  Manfred Stock
 * @version $Revision: 883 $
 */
class Hashtable {
  // The total number of entries in the hash table.
  private var count: Int = 0
  
  // Initial capacity of the hash, which grows a needed
  private val initialCapacity = 101

  // The load factor for the hashtable.
  private var loadFactor : Float = 0.75f

  //  Rehashes the table when count exceeds this threshold.
  private var threshold : Int = (initialCapacity * loadFactor).asInstanceOf[Int]
    
  //  The hash table data.
  private var table : Array[List[HashtableEntry]] = new Array[List[HashtableEntry]](initialCapacity)
    
  /**
   * Gets the record associated with the specified object.
   * System.identityHashCode() is used as key.
   * @param	o The object to search for
   * @return	the element for the object or <code>None</code> if the
   *    			object is not defined in the hash table.
   */
  def get (o: Any) : Option[HashtableEntry] = synchronized {
    val hash = System.identityHashCode(o)
    val e = table((hash & 0x7FFFFFFF) % table.length)
    if (e != null) {
      e.find(el => el.hash == hash && el.holds(o))
    }
    else {
      None
    }
  }

  /**
   * Gets the record that uses the specified UrtWeakReference
   * to link to the object it is associated with.
   * @param	r The UrtWeakReference to search for
   * @return	the element for the object or null if the
   *          object is not defined in the hash table.
   */
  def get(r: UrtWeakReference) : Option[HashtableEntry] = synchronized {
    val e = table((r.hash & 0x7FFFFFFF) % table.length)
    if (e != null) {
      e.find(el => el.hash == r.hash && el.obj == r)  
    }
    else {
      None
    }
  }

  /**
   * Rehashes the content of the table into a bigger table.
   * This method is called automatically when the hashtable's
   * size exceeds the threshold.
   */
  private def rehash {
    val newCapacity = table.length * 2 + 1
    val newTable = new Array[List[HashtableEntry]](newCapacity)
    threshold = (newCapacity * loadFactor).asInstanceOf[Int]
    table.reverse.foreach { e0 => 
      if (e0 != null) {
        e0.foreach { e1 =>  
          val index = (e1.hash & 0x7FFFFFFF) % newCapacity
          if (newTable(index) != null) {
            newTable(index) = e1 :: newTable(index)
          }
          else {
            newTable(index) = List(e1)
          }
        }
      }
    }
    table = newTable
  }

  /**
   * Puts the specified element into the hashtable, using
   * System.identityHashCode() as key.
   * The element and the entry cannot be null.
   *
   * @param	o the element to add
   * @param	e the entry to use for this element
   *
   * @return	the entry in the hashtable for this object
   *
   * @exception	NullPointerException If the object or the
   *            entry is equal to null.
   */
  def put (obj: AnyRef, owner: HashtableEntry) : HashtableEntry = synchronized {
    // Make sure the entry is not null.
    if ( (obj == null) || (owner == null) ) {
      throw new NullPointerException()
    }

    // scdaniel: do this before the loop, saves some time
    if ( count >= threshold ) {
      // Rehash the table if the threshold is exceeded.
      rehash
    }
		
    val newEntry = obj match {
      case thread: Thread => new HashtableThreadEntry(new UrtWeakReference(thread))
      case _              => new HashtableEntry(new UrtWeakReference(obj))
    }
    newEntry.owner = owner
    
    val index = (newEntry.hash & 0x7FFFFFFF) % table.length
    val entry = table(index)
    def insert = {
      table(index) = newEntry :: table(index)
      count += 1
      newEntry
    }
    if (entry != null) {
      entry.find(i => (i.hash == newEntry.hash) && i.holds(obj)) match {
        case Some(value) => value
        case None        => insert
      }
    } 
    else {
      table(index) = Nil
      insert
    }
  }

  /**
	 * Removes the specified element.
   * Does nothing if the key is not present.
   *
   * @param	e the entry that needs to be removed
   */
  def remove (e : HashtableEntry) = synchronized {
    val index = (e.hash & 0x7FFFFFFF) % table.length
    val i = table(index)
    if (i != null) {
      table(index) = i.remove(entry => if (entry == e) { count -= 1; true } else { false })
    }
  }
  
  /**
   * Convert hashtable to a string.
   * @return the table as a string.
   */
  override def toString : String = {
    ((table filter (l => l != null && !l.isEmpty)) map 
        {l => ((l.head.hash & 0x7FFFFFFF) % table.length)+"\t => "+l}
    ) mkString ("","\n","") 
  }
}

/**
 * The class for the entries used in Hashtable.
 * 
 * The fields are accessible directly for speed reasons.
 * @param obj Object contained in this hashtable entry.
 * @param owner Owner of the aforementioned object.
 * 
 * @author  Manfred Stock
 * @version $Revision: 883 $
 */
class HashtableEntry(var obj: UrtWeakReference) {
  var owner : HashtableEntry = _
  val hash = System.identityHashCode(obj.get)
  
  /*
   *	Attributes for ownerHashtable
   */
  var children = 0
    	
  /**
   *	Tests if this instance of HashtableEntry holds
   *	the Object specified. (ie. links to it)
   * 
   *	@param	o	the Object to test
   *	@return	true if this HashtableEntry holds the specified Object
   */
  def holds(o: Any) : Boolean = {
  	this.obj.get() == o
  }
	
  /**
   * Convert hashtable entry to a string.
   * @return the string.
   */
  override def toString : String = {
    val self = if (this.obj  != null) this.obj.get  else this.obj
    val ow   = if (owner.obj != null) owner.obj.get else owner.obj
    hash+" -> "+self+" ("+ow+")"
  }
}

/**
 * This class is used for entries associated to threads.
 * Provides additional information to allow to set the owner in the constructor
 * and therefore associate objects created in the constructor (and fields) with
 * their correct owners.
 * This wouldn't work when setting the owner after the "new" operator.
 * 
 * The fields are accessible directly for speed reasons.
 * 
 * @param thread	The thread which should be contained in the entry.
 * 
 * @author  Manfred Stock
 * @version $Revision: 883 $
 */
class HashtableThreadEntry(var thread : UrtWeakReference) 
    extends HashtableEntry(thread) {
  var currentObject : AnyRef = _
  var objectClass : Object = _
  var modifier = Modifiers.PEER
}

/**
 * A class extending WeakReference and adding the additional attribute
 * hash storing the <code>System.identityHashCode</code> of the object it 
 * refers to. This allows to delete the hashtable entry of that object if
 * the object is collected. We are also notified by the reference queue.
 * 
 * @author  Manfred Stock
 * @version $Revision: 883 $
 */
class UrtWeakReference(o: AnyRef, q: ReferenceQueue[AnyRef]) extends WeakReference[AnyRef](o,q) {
    // the hashcode of the object we're linking to
    val hash : Int = System.identityHashCode(o) 
	
    // just calls super and sets the hashcode
    def this(o: AnyRef) {
      this(o,null)
    }
}