/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * 
 * $Id: UTSRuntime.scala 817 2008-01-25 19:44:05Z ms $
 */
package ch.ethz.inf.sct.uts.rt

import ch.ethz.inf.sct.uts.rt.implementation._
import ch.ethz.inf.sct.uts.rt.policy._

/**
 * Runtime for the Universe Type System checks. Instantiates an object
 * of the class specified on the JVM command-line switch -DUrtImplementation=<class> 
 * and stores it into the static field <code>handler</code> and also
 * instiantiates an object of the class specified by -DurtPolicy=<class> which is stored in
 * the static field <code>policy</code>.
 * Defaults to <code>implementation.UTSRuntimeImplementationDefault</code> and 
 * <code>policy.UTSDefaultPolicy</code> if nothing is specified or instantiation fails.
 * 
 * @author  Manfred Stock
 * @version $Revision: 817 $
 */
object UTSRuntime {
  /**
   * Runtime reference to an instance of a class implementing <code>UTSRuntimeBase</code>.
	 */
	val handler : ImplementationBase = {
    /* get name of implementation class */
    val runtimeClass = System.getProperty("UrtImplementation")
    if (runtimeClass != null) {
      /* try to create an instance of this class */
      try {
        Class.forName(runtimeClass).newInstance().asInstanceOf[ImplementationBase]
      } 
      catch {
        case e : Exception => 
          System.err.println("WARNING: the specified Universe Type System Runtime Implementation Class (" +runtimeClass +") could not be instantiated, using the default instead.")
          new DefaultImplementation 
      }
    }
    else {
      new DefaultImplementation
    }
  }

  /**
   * Reference to an instance of a class used as policy for the runtime checks.
   */
  val policy : PolicyBase = {
    /* get name of policy class */
    val runtimeClass = System.getProperty("UrtPolicy");
    if (runtimeClass != null) {
      /* try to create an instance of this class */
      try {
        Class.forName(runtimeClass).newInstance().asInstanceOf[PolicyBase]
      } 
      catch {
        case e: Exception => 
          System.err.println("WARNING: the specified Universe Type System Runtime Policy Class (" +runtimeClass+") could not be instantiated, using the default instead.")
          new DefaultPolicy
      }
    }
    else {
      new DefaultPolicy();
    }
  }
}