/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * 
 * $Id: UTSMethodSignature.scala 883 2008-02-01 18:59:56Z ms $
 */
package ch.ethz.inf.sct.uts.plugin.staticcheck.common;

import scala.tools.nsc._
import ch.ethz.inf.sct.uts.plugin.common._
import ch.ethz.inf.sct.uts.plugin.staticcheck._
import ch.ethz.inf.sct.uts.annotation._

/**
 * Trait to encapsulate a <code>MethodSignature</code> representation and some factory 
 * and helper methods suitable for method handling in the Universe type system.
 * 
 * @author  Manfred Stock
 * @version $Revision: 883 $
 */
trait UTSMethodSignature[G <: Global] {
  self: TypeAbstraction[G] => 
   
  import global._
  import UTSDefaults._
  
  /**
   * Type for describing method signatures.
   * @param typeParams Type parameters of this method, if any.
   * @param isPure     Flag if this method is pure.
   * @param returnType Return type of this method.
   * @param paramTypes Method parameter types, if any.
   * @param sym        Symbol of the method. 
   */
  abstract class MethodSignature(val typeParams: List[UType], val isPure: Boolean, val returnType: UType, val paramTypes: List[UType], val sym: Symbol) {
    /**
     * Check equality of two method signatures, taking the purity into account.
     * @param that The other method signature.
     * @return if the signatures match.
     */
    def =:= (that: MethodSignature) : Boolean = {        
      (this =~= that) &&
        (this.isPure == that.isPure)
    }

    /**
     * Check equality of two method signatures, without taking the purity into account.
     * @param  that The other method signature.
     * @return if the signatures match.
     */
    def =~= (that: MethodSignature) : Boolean = {
      this =~= (that,(a,b) => a =:= b)
    }
      
    /**
     * Check equality of two method signatures, without taking the purity into account. Uses 
     * a comparator function which is used to compare the return types. If both methods are
     * constructors, the return types are allowed to be different.
     * @param that The other method signature.
     * @param cmp  Function to compare the return type of the methods.
     * @return if the signatures match.
     */
    def =~= (that: MethodSignature, cmp: (UType, UType) => Boolean) : Boolean = {
      ((cmp(this.returnType, that.returnType)) || (this.sym.isConstructor && that.sym.isConstructor)) &&
        (List.forall2(this.typeParams, that.typeParams)((a,b) => a =:= b)) &&
          (List.forall2(this.paramTypes, that.paramTypes)((a,b) => a =:= b))
    }
   
    /**
     * Check if method in <code>this</code> correctly overrides method in <code>that</code>.
     * @param that The method which should be overriden by the method in <code>this</code>.
     * @return if overriding is correct.
     */
    def overrides (that: MethodSignature) : Boolean
    
    /**
     * Replace type parameters in parameter types by actual type arguments.
     * @param from The <code>NType</code> from which the actual arguments can be retrieved.
     */
    def resolveTypeVarsFrom(from: NType) : MethodSignature = {
      methodSignature(
          typeParams,
          isPure,
          returnType,
          paramTypes.map{
            case nt: NType         => nt
            case tv: TVarID        => from.tpe match {
              case tr@TypeRef(pre,sym,args) => UType(tr.transform(tv.tpe))
              case _                        => UType(tv.tpe.asSeenFrom(from.tpe, sym.owner))
            }
            case et: ErroneousType => et
          },
          sym
      )
    }  

    /**
     * Substitute type variables by actual type arguments. If the replacement of a type 
     * parameter is still a typevariable, go to its upper bound.
     * @param targs List of actual type arguments.
     * @param args List of actual arguments.
     */
    def substTypeArgs(targs: List[UType],args: List[UType]) : MethodSignature = {
      val map = ((typeParams map {_.tpe}) zip targs)
       
      def subst(on: UType) : UType = {
        on match {
          case nt: NType         => nt.subst(map)
          case tv: TVarID        => getTypeFromMapping(map,tv)
          case et: ErroneousType => et
        }
      }
      methodSignature(
          (typeParams map subst) map {
            case tv: TVarID => tv.ubgamma
            case t:  UType  => t
          },
          isPure,
          subst(returnType),
          paramTypes map subst,
          sym
      )
    }
    
    /**
     * Convert this method signature to a string.
     * @return the string representation of this method signature.
     */
    override def toString : String = {
      {if (isPure) "pure " else ""} +
        "def " + sym.nameString +
          {if (typeParams != Nil) typeParams.mkString("[",",","]") else ""} +
            paramTypes.mkString("(",",",")") + ": " + returnType
    }
  }  

  /**
   * Factory method to create a new method signature.
   * @param sym Compiler symbol of the method.
   * @return the method signature if it was initialized, or a reason why this failed.
   */
  def method(sym: Symbol) : ROption[MethodSignature] = {
    if (sym.isMethod) {
      val mtype = sym.tpe
      val returnType = UType(mtype.resultType) match {
        case et: ErroneousType => et
        case t                 => t
      }
      var typeParams = mtype.typeParams.map(tp => UType(tp.tpe))
      var paramTypes = mtype.paramTypes.map(pt => UType(pt))

      RSome(methodSignature(typeParams,isMethodPureAnnotated(sym),returnType,paramTypes,sym))
    }
    else {
      RNone("Given symbol "+sym+" is not a method.")
    }
  }

  
  /**
   * Factory method to create new <code>MethodSignature</code> instances.
   * @param typeParams Type parameters of this method, if any.
   * @param isPure     Flag if this method is pure.
   * @param returnType Return type of this method.
   * @param paramTypes Method parameter types, if any.
   * @param sym        Symbol of the method.
   */
  def methodSignature(typeParams: List[UType], isPure: Boolean, returnType: UType, paramTypes: List[UType], sym: Symbol) : MethodSignature

  /**
   * Check if a given method symbol is annotated as being pure.
   * @param sym Method symbol which could be annotated as being pure.
   * @return if the method symbol is annotated as being pure.
   */
  def isMethodPureAnnotated(sym: Symbol) : Boolean = {
    sym.isMethod && sym.attributes.map(_.atp.toLongString).exists(_.matches(pure().getClass.getName))
  }
     
  /**
   * Check if a given method may be assumed as being pure from the defaults given 
   * in <code>UTSDefaults</code>.
   * @param sym Method symbol of the method which should be checked.
   * @return if the method may be assumed as being pure.
   */
  def methodAssumableAsPure(sym: Symbol) : Boolean = {
    if (sym.isMethod) {
      var relevantNames : List[String] = Nil
      // Get base classes which may contain "pure" methods, and add the symbol names 
      // of these methods to the relevantNames list
      val relevantBaseClasses = sym.owner.tpe.baseClasses.filter(bc => pureMethods.get(bc.fullNameString) match {
        case Some(list) => list.foreach(m => relevantNames = bc.fullNameString+"."+m :: relevantNames); true
        case None       => false
      })
         
      val overriddenSymbols = relevantBaseClasses.map(sym.overriddenSymbol(_)).filter(s => s != NoSymbol)

      overriddenSymbols.exists(s => relevantNames.contains(s.fullNameString))
    }
    else {
      false
    }
  }
}
