/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * 
 * $Id: TypeRules.scala 883 2008-02-01 18:59:56Z ms $
 */
package ch.ethz.inf.sct.uts.plugin.staticcheck

import scala.tools.nsc._
import ch.ethz.inf.sct.uts.annotation._
import ch.ethz.inf.sct.uts.plugin.common._
import ch.ethz.inf.sct.uts.plugin.staticcheck.common._

/**
 * Class providing the interface for type rules of a Universe type system.
 * @param global An instance of <code>Global</code>.
 * @param logger The logger to use for logging.
 * 
 * @author  Manfred Stock
 * @version $Revision: 883 $
 */
abstract class TypeRules[G <: Global](val global: G, val logger: UTSLogger) extends TypeAbstraction[G] {
  import global._
  import UTSDefaults._  
  import extendedType._
  
  /**
   * Process options which were not accepted by the plugin itself and 
   * may therefore be meant for the used typerules.
   * @param o The option.
   * @return if the option was accepted.
   */
  def processOption(o: String) : Boolean
      
  /**
   * Get help about the options implemented by the given typerules.
   * @param name Name of the plugin which got the option.
   * @return the string with help on the options.
   */
  def getOptionsHelp(name: String) : String
  
  /**
   * Get a list with information about the selected options.
   * @return a list with descriptions about the selected options.
   */
  def getActiveOptions : List[String]
  
  /**
   * Field lookup to get the type of a field of a nonvariable type, viewpoint adapted to it.
   * @param c Nonvariable type which is supposed to contain the field <code>f</code>.
   * @param f Field one is looking for.
   * @return the type of the field adapted to <code>nt</code> if found.
   */
  def fType(nt: NType, f: Symbol) : UType
      
  /**
   * Get the type of a method.
   * @param c The type which should contain the method.
   * @param m The method symbol as found in the tree.
   * @return the type of the method. 
   */
  def mType(c: Type, m: Symbol) : ROption[MethodSignature]
    
  /**
   * Get the type of a method, adapt it to a nonvariable type.
   * @param n The type which should contain the method, and the signature should be adapted to.
   * @param m The method symbol as found in the tree.
   * @return the type of the method. 
   */
  def mType(n: NType,m: Symbol) : ROption[MethodSignature]
  
  /**
   * Check if a given method of a class overrides correctly.
   * @param m Method to check.
   * @return if the given method successfully overrides the one in the parent class.
   */
  def ovrride(m: Symbol) : ROption[Boolean]
  
  /**
   * Check if an assignment is ok, possibly respecting the type of the owner 
   * of the reference some value is assigned to.
   * @param vtype           The <code>UType</code> of the field which some value gets assigned to.
   * @param lhsRefOwnerType The <code>UType</code> of the object which contains the reference to the field.
   * @param atype           The <code>UType</code> of the value which is assigned.
   * @return the type of field which got assigned a value.
   */
  def checkAssignment(vtype: UType, lhsRefOwnerType: UType, atype: UType) : UType

  /**
   * Check if an assignment is ok concerning the types.
   * @param lt The <code>UType</code> which some value would be assigned to.
   * @param rt The <code>UType</code> which would be assigned.
   * @return the type of lt.
   */
  def checkAssignment(lt: UType, rt: UType) : UType
 
  /**
   * Check the invocation of a method (GT-Invk).
   * @param receiver Receiver of the method call.
   * @param method   Symbol of the called method.
   * @param args     Arguments to the method call.
   * @param targs    Type arguments to the call which should be checked.
   * @return the return type of the method call, if any.
   */
  def checkInvocation(receiver: UType, method: Symbol, args: List[UType], targs: List[UType]) : UType

  /**
   * Check if a given Type can be instantiated.
   * @param t Type to be checked.
   */
  def checkNew(t: UType) : UType
  
  /**
   * Get the type of a field read.
   * @param nt Class which contains the field.
   * @param f  Symbol of the field.
   * @return the type of the field. 
   */
  def fieldRead(nt: NType, f: Symbol) : UType
  
  /**
   * Check a class definition.
   * @param cls    Symbol of the class.
   * @param params Type parameters.
   * @return the list of parameters, possibly containing <code>InvalidType</code>
   *         instances if a parameter was invalid.
   */
  def checkClassDefinition(cls: Symbol, params: List[(util.Position,UType)]) : List[UType]
  
  /**
   * Check the definition of a value, ie. field or local variable.
   * @param sym    Symbol which is defined.
   * @param rhstpe Type of the initializer.
   * @return the type of the defined field.
   */
  def checkValueDefinition(sym: Symbol, rhstpe: UType) : UType
   
  /**
   * Check a method definition.
   * @param method   The symbol of the defined method.
   * @param name     Name of the method.
   * @param tparams  Type parameters of the method.
   * @param vparamss Value parameter types.
   * @param frtype   Formal return type.
   * @param artype   Actual return type.
   */
  def checkMethodDefinition(method: Symbol, name: String, tparams: List[UType], vparamss: List[List[UType]],frtype: UType, artype: UType) : UType
  
  /**
   * Check <code>if</code>-<code>then</code>-<code>else</code> statement.
   * @param iftpe     Type of the if expression.
   * @param cond      Type of the condition.
   * @param then      Type of the <code>then</code> branch.
   * @param thenpos   Position of then block.
   * @param otherwise Type of the <code>else</code> branch.
   * @param elsepos   Position of else block.
   * @return the type of the if-then-else statement.
   */
  def checkIfThenElse(iftpe: UType, cond: UType, then: UType, thenpos: util.Position, otherwise: UType, elsepos: util.Position) : UType
  
  /**
   * Check a pattern <code>match</code>.
   * @param matchtype Type of the match expression as given by the Scala compiler.
   * @param casetypes Types of the different case statements.
   * @return the type of the pattern match.
   */
  def checkMatch(matchtype: UType, casetypes: List[(util.Position,UType)]) : UType
  
  /**
   * Check a <code>try</code> statement. 
   * @param trytype       Type of the try statement.
   * @param blocktype     Type of the block in the try statement.
   * @param catchestypes  Types of the <code>catch</code> statements.
   * @param finalizertype Type of the <code>finalize</code> statement.
   * @return the type of the <code>try</code> statement.
   */
  def checkTry(trytype: UType, blocktype: UType, blockpos: util.Position, catchestypes: List[(util.Position,UType)],finalizertype: UType) : UType
  
  /**
   * Check a <code>this</code> access.
   * @param tpe Type of <code>this</code>
   * @return the possibly modified type of this.
   */
  def checkThis(tpe: UType) : UType
  
  /**
   * Check a field read.
   * @param target Type of the target on which the select takes place.
   * @param field  The field which is selected.
   * @return the type of the field, after viewpoint adaptation etc.
   */
  def checkSelect(target: UType, field: Symbol) : UType

  /**
   * Check a typed expression (ie. something like expr: tpt
   * @param expr Type of the expression.
   * @param tpt  Type the expression should have.
   */
  def checkTyped(expr: UType, tpt: UType) : UType  
}