/*
 * Copyright (C) 2000-2003 Iowa State University
 *
 * This file is part of mjc, the MultiJava Compiler.
 *
 * based in part on work:
 *
 * Copyright (C) 1990-99 DMS Decision Management Systems Ges.m.b.H.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * $Id: Utils.java,v 1.21 2007/12/20 12:25:26 chalin Exp $
 */

package org.multijava.util;

import java.util.Vector;
import java.io.File;
import java.io.IOException;
import java.util.LinkedList;
import java.util.List;
import java.util.Iterator;

/**
 * This class defines severals utilities methods used in source code
 */
public abstract class Utils implements Cloneable {

    // ----------------------------------------------------------------------
    // UTILITIES
    // ----------------------------------------------------------------------

    /**
     * Check if an assertion is valid
     *
     * @exception	RuntimeException	the entire token reference
     */
    public static final void assertTrue(boolean b, String message) {
	if (DEBUG_ENABLED) {
	    if (!b) {
		RuntimeException re = new RuntimeException(message);
		re.printStackTrace();
		if (debugLevel > DBG_LEVEL_NO) {
		    throw re;
		}
	    }
	}
    }

    /**
     * Throws an error.
     *
     * @exception	RuntimeException	the entire token reference
     */
    public static final void fail(String message) {
	RuntimeException re = new RuntimeException(message);
	re.printStackTrace();
	if (debugLevel > DBG_LEVEL_NO) {
	    throw re;
	}
    }

    /**
     * Check if an assertion is valid
     *
     * @exception	RuntimeException	the entire token reference
     */
    public static final void assertTrue(boolean b) {
	if (DEBUG_ENABLED) {
	    if (!b) {
		RuntimeException re = new RuntimeException("assertion failed");
		re.printStackTrace();
		if (debugLevel > DBG_LEVEL_NO) {
		    throw re;
		}
	    }
	}
    }

    /**
     * Throws an error.
     *
     * @exception	RuntimeException	the entire token reference
     */
    // This should be here, but causes to many other failures right now.
    //x@ assignable System.out.output;
    public static final void fail() {
	RuntimeException re = new RuntimeException("fail reached");
	re.printStackTrace();
	if (debugLevel > DBG_LEVEL_NO) {
	    throw re;
	}
    }

    // ----------------------------------------------------------------------
    // UTILITIES
    // ----------------------------------------------------------------------

    /**
     * Returns true if any of the bits set in <code>flags</code> are 
     * are also enabled in <code>modifiers</code>.  This method is 
     * usually used to check for a single flag.
     *
     * @param modifiers the modifiers bit mask
     * @param flags the flags to be checked
     * @return (modifiers & flags) != 0 */
    public static /*@ pure @*/ boolean hasFlag(long modifiers, long flags) {
	return (modifiers & flags) != 0;
    }

    /**
     * Returns true if <code>modifiers</code> has any bits set other
     * than those in <code>flags</code>.
     *
     * @param modifiers the modifiers bit mask
     * @param flags the flags that are supposed to be in <code>modifiers</code>
     * @return (modifiers & flags) != modifiers */
    public static /*@ pure @*/ boolean hasOtherFlags(long modifiers, long flags) {
	return (modifiers & flags) != modifiers;
    }

    /**
     * Returns the <code>modifiers</code> with the non-Java bit masks stripped.
     *
     * @param modifiers the modifiers bit mask
     * @return modifiers & 0x0000FFFF */
    public static /*@ pure @*/ long stripNonJavaModifiers(long modifiers) {
	// 16 low-order bits (last 4 hex digits) are used for VM flags
	return (modifiers & 0x000000000000FFFFL);
    }

    /**
     * Returns the <code>modifiers</code> with the Java bit masks stripped.
     *
     * @param modifiers the modifiers bit mask
     * @return modifiers & 0xFFFFFFFFFFFF0000L */
    public static /*@ pure @*/ long stripJavaModifiers(long modifiers) {
	// 16 low-order bits (last 4 hex digits) are used for VM flags
	return (modifiers & 0xFFFFFFFFFFFF0000L);
    }

    /**
     * Returns the <code>modifiers</code> with the Private modifier stripped.
     *
     * @param modifiers the modifiers bit mask
     * @return modifiers & 0xFFFFFFFFFFFFFFFDL */
    public static /*@ pure @*/ long stripPrivateModifier(long modifiers) {
	// 16 low-order bits (last 4 hex digits) are used for VM flags
	return (modifiers & 0xFFFFFFFFFFFFFFFDL);
    }

    
    /**
     * Creates a typed array from a vector.
     * @param	vect		the vector containing the elements
     * @param	type		the type of the elements
     */
    public static Object[] vectorToArray(Vector vect, Class type) {
	if (vect != null && vect.size() > 0) {
	    Object[]	array = (Object[])java.lang.reflect.Array.newInstance(type, vect.size());
      
	    try {
		vect.copyInto(array);
	    } catch (ArrayStoreException e) {
		System.err.println("Array was:" + vect.elementAt(0));
		System.err.println("New type :" + array.getClass());
		throw e;
	    }
	    return array;
	} else {
	    return (Object[])java.lang.reflect.Array.newInstance(type, 0);
	}
    }
  
    /**
     * Creates a int array from a vector.
     * @param	vect		the vector containing the elements
     */
    public static int[] vectorToIntArray(Vector vect) {
	if (vect != null && vect.size() > 0) {
	    int[]	array = new int[vect.size()];
      
	    for (int i = array.length - 1; i >= 0; i--) {
		array[i] = ((Integer)vect.elementAt(i)).intValue();
	    }

	    return array;
	} else {
	    return new int[]{}; // $$$ static ?
	}
    }
    
    /** Combines two reference arrays of into one; the result has the same dynamic type
	as the first argument.
    */
    public static Object[] combineArrays(Object[] x, Object[] y) {
	if (x == null) return y;
	if (y == null) return x;
	Object[] res = (Object[])java.lang.reflect.Array.newInstance(x.getClass().getComponentType(),x.length+y.length);
	System.arraycopy(x,0,res,0,x.length);
	System.arraycopy(y,0,res,x.length,y.length);
	return res;
    }

    /**
     * Splits a string like:
     *   "java/lang/System/out"
     * into two strings:
     *    "java/lang/System" and "out"
     */
    public static String[] splitQualifiedName(String name, char separator) {
	String[]	result = new String[2];
	int		pos;

	pos = name.lastIndexOf(separator);

	if (pos == -1) {
	    // no '/' in string
	    result[0] = "";
	    result[1] = name;
	} else {
	    result[0] = name.substring(0, pos);
	    result[1] = name.substring(pos + 1);
	}

	return result;
    }

    /**
     * Splits a string like:
     *   "java/lang/System/out"
     * into two strings:
     *    "java/lang/System" and "out"
     */
    public static String[] splitQualifiedName(String name) {
	return splitQualifiedName(name, '/');
    }

    /**
     * Returns a String formed by translating Java escape sequences in
     * the given String into single Unicode characters.  The escape
     * sequences are those of JLS2, 3.10.6 with the addition of
     * Unicode escapes.  
     *
     * @see #escapeString(String)
     */
    public static final String unescapeString( String val ) {
	char[] buffer = new char[val.length()];
	char[] old = val.toCharArray();
	if (old.length > buffer.length) {
	    buffer = new char[ old.length ];
	} // end of if 
	
	int pos = 0;
	int outPos = 0;
	while (pos < old.length) {
	    char out;
	    if (old[pos] != '\\') {
		// not an escape sequence
		out = old[pos];
		pos++;
	    } else {
		    // keep track of whether the first digit in an octal escape
		    // sequence is between 0 and 3.
		boolean zeroToThree = false;
		// start of an escape sequence
		switch (old[pos+1]) {
		case '0':
		case '1':
		case '2':
		case '3': zeroToThree = true;
		case '4':
	        case '5':
		case '6':
		case '7': {
		    String octal;
		    // octal escape
		    if (pos+2 < old.length &&
			old[pos+2] >= '0' && old[pos+2] <= '7') {
			if (zeroToThree && pos+3 < old.length &&
			    old[pos+3] >= '0' && old[pos+3] <= '7') {
			    octal = new String( old, pos+1, 3 );
			    pos += 4; // skip \ooo
			} else {
			    octal = new String( old, pos+1, 2 );
			    pos += 3; // skip \oo
			}
		    } else {
			octal = new String( old, pos+1, 1 );
			pos += 2; // skip \o
		    }
		    out = (char) Integer.parseInt( octal, 8 );
		    break;
		}
		  
		case 'u':
		    // unicode escape
		    String unicode = new String( old, pos+2, 4 );
		    out = (char) Integer.parseInt( unicode, 16 );
		    pos += 6; // skip \ u xxxx
		    break;
		    
		case 'n':
		    out = (char) 10;
		    pos += 2;
		    break;
			
		case 'r':
		    out = (char) 13;
		    pos += 2;
		    break;
			
		case 't':
		    out = (char) 9;
		    pos += 2;
		    break;
			
		case 'b':
		    out = (char) 8;
		    pos += 2;
		    break;
			
		case 'f':
		    out = (char) 12;
		    pos += 2;
		    break;
			
		case '\'':
		case '\"':
		case '\\':
		    out = old[ pos+1 ];
		    pos += 2;
		    break;

		default:
		    throw new IllegalArgumentException( "bad escape sequence in " + val );
		} // end of switch
	    } // end of if	
	    buffer[outPos++] = out;
	} // end of while
	return new String( buffer, 0, outPos );
    }

    /**
     * Returns a String formed by translating Unicode characters for
     * the standard Java escape sequences into the corresponding
     * escape sequences from JLS2, 3.10.6.  Octal and unicode escapes
     * are not generated.
     *
     * @see #escapeString(String,boolean)
     * @see #unescapeString(String) */
    public static final String escapeString( String val ) {
	return escapeString( val, true );
    }
    /**
     * Returns a String formed by translating Unicode characters for
     * the standard Java escape sequences into the corresponding
     * escape sequences from JLS2, 3.10.6.  Octal and unicode escapes
     * are not generated.
     *
     * @see #unescapeString(String) */
    public static final String escapeString( String val, 
					     boolean escapeApostrophes ) {
	// escaped characters can at most double the length
	char[] buffer = new char[ val.length() * 2 ];
	char[] old = val.toCharArray();
	int pos = 0;
	for (int i = 0; i < old.length; i++) {
	    switch (old[i]) {
	    case '\b':
		buffer[pos++] = '\\';
		buffer[pos++] = 'b';
		break;
		
	    case '\t':
		buffer[pos++] = '\\';
		buffer[pos++] = 't';
		break;
		
	    case '\n':
		buffer[pos++] = '\\';
		buffer[pos++] = 'n';
		break;
		
	    case '\f':
		buffer[pos++] = '\\';
		buffer[pos++] = 'f';
		break;
		
	    case '\r':
		buffer[pos++] = '\\';
		buffer[pos++] = 'r';
		break;
		
	    case '"':
		buffer[pos++] = '\\';
		buffer[pos++] = '"';
		break;
		
	    case '\'':
		if (escapeApostrophes) {
		    buffer[pos++] = '\\';
		    buffer[pos++] = '\'';
		} else {
		    buffer[pos++] = old[i];
		} // end of else
		break;
		
	    case '\\':
		buffer[pos++] = '\\';
		buffer[pos++] = '\\';
		break;
		
	    default:
		buffer[pos++] = old[i];
		break;
	    } // end of switch
	} // end of for
	return new String( buffer, 0, pos );
    }


    /**
     * Returns a String representing the relative path from the
     * current user directory, <CODE>user.dir</CODE>, to the given
     * <CODE>file</CODE>.
     *
     * <pre><jml>
     * requires file != null;
     * </jml></pre>
     */
    public static final String relativePathTo(File file) {

	File user = 
	    new File(System.getProperty("user.dir")).getAbsoluteFile();
	File reference= file.isAbsolute() ? file : file.getAbsoluteFile();

	try {
	    user = user.getCanonicalFile();
	} catch(IOException e) {
	    // ignores exception since if it is thrown the absolute
	    // path is the best answer we can get
	}
	try {
	    reference = reference.getCanonicalFile();
	} catch(IOException e) {
	    // ignores exception since if it is thrown the absolute
	    // path is the best answer we can get
	}

	List userParts = parsePathParts(user);
	List referenceParts = parsePathParts(reference);

	while (userParts.size() > 0 && referenceParts.size() > 0) {
	    if ((caseInsensitiveFS 
		 && !((String) userParts.get(0)).
		 equalsIgnoreCase((String) referenceParts.get(0)))
		|| (!caseInsensitiveFS
		    && !userParts.get(0).equals(referenceParts.get(0)))) {
		break;
	    }
	    userParts.remove(0);
	    referenceParts.remove(0);
	}

	StringBuffer result = new StringBuffer();
	// need to back out of the mismatches in userParts
	while (userParts.size() > 0) {
	    result.append(FILE_SEP).append(UP);
	    userParts.remove(0);
	}
	
	// need to move forward through the mismatches in referenceParts
	while (referenceParts.size() > 0) {
	    // first is really a String, but we save a cast by using Object
	    Object first = referenceParts.remove(0);
	    result.append(FILE_SEP).append(first);
	}
	
	result.delete(0, FILE_SEP.length());
	
	if (result.length() == 0) {
	    return CURRENT;
	}

	return result.toString();
    }


    /**
     * Returns a list of Strings representing the parts of the given
     * path, assembled by repeatedly invoking <CODE>getName()</CODE>
     * on <CODE>getParentFile()</CODE> of the given <CODE>path</CODE>
     * until no parent path parts remain.  */
    public static final List parsePathParts(File path) {
	LinkedList result = new LinkedList();
	while (path != null) {
	    if (!path.getName().equals(CURRENT)) {
		result.addFirst(path.getName());
	    }
	    path = path.getParentFile();
	}
	return result;
    }

    /**
     * Returns the canonical path of the given File, or the absolute
     * path if an exception is thrown by the underlying file system
     * when constructing the canonical path.
     */
    public static String getFilePath(File f) {
	String key;
	try {
	    key = f.getCanonicalPath();
	} catch (Exception e) {
	    key = f.getAbsolutePath();
	}
	return key;
    }

    // ----------------------------------------------------------------------
    // PUBLIC CONSTANTS
    // ----------------------------------------------------------------------

    public static final int	DBG_LEVEL_NO	= 0;
    public static final int	DBG_LEVEL_LOW	= 1;
    public static final int	DBG_LEVEL_HIGH	= 2;

    private static String CURRENT = ".";
    private static String UP = "..";
    private static String FILE_SEP = System.getProperty("file.separator");

    private static boolean caseInsensitiveFS =
	FILE_SEP.equals("\\")	// Only Windows would use backslash
	|| (System.getProperty("os.name").indexOf("Mac") >= 0);

    // ----------------------------------------------------------------------
    // PRIVATE DATA MEMBERS
    // ----------------------------------------------------------------------

    private static final boolean	DEBUG_ENABLED = true;

    private static int		debugLevel	= DBG_LEVEL_HIGH;
}
