/*
 * This file is part the Universe Runtime Classes.
 *
 * Copyright (C) 2003-2005 Swiss Federal Institute of Technology Zurich
 *
 * Part of mjc, the MultiJava Compiler.
 *
 * Copyright (C) 2000-2005 Iowa State University
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: UrtVisualizer.java,v 1.1 2005/06/12 19:38:28 wdietl Exp $
 */

package org.multijava.universes.rt.impl;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;

import org.multijava.universes.rt.UrtImplementation;

/**
 * Alternative implementation of UrtDefaultImplementation that generates
 * dot code to generate a visualization of the ownership relations.
 * 
 * It prints the code to the file specified in the Java property "UrtOutfile".
 * The language used to describe the graph is "dot", which is part of the
 * graphviz package.
 * @see http://www.graphviz.org/
 * 
 * The name of the objects can be modified by changing the getName() function.
 * Standard is "uniqueNumber - shortClassName@identityHashcode".
 * The unique number is a counter lets you see the order in which objects were
 * created.
 * 
 * To automate this whole thing just put the something like the following lines
 * in your Makefile:
 * 
 *	visualize: compile
 * 	   java -DUniverseRuntime=org.multijava.universes.rt.UrtVisualizer \
 * 			-DUrtOutfile=dot.dot helloWorld
 *	   dot -o dot.ps -Tps dot.dot
 *	   gv dot.ps &
 * 
 * @author scdaniel
 */
public class UrtVisualizer
    extends UrtDefaultImplementation
    implements UrtImplementation {
	
    /**
     * Objects needed to print to a file.
     */
    FileWriter fw;
    PrintWriter pw;

    // the counter
    int counter = 0;
	
    /**
     * Opens the outfile and prints the information that has already been
     * collected by functions called in the constructor of the superclasses.
     * 
     * @throws IOException
     */
    public UrtVisualizer () throws IOException {
	/* open file and start graph */
	String outfile = System.getProperty("UrtOutfile");

	if ( outfile == null ) {
	    throw new IOException(UrtVisualizer.class.toString() +
				  ": no outputfile defined!");
	}

	fw = new FileWriter(new File(outfile));
	pw = new PrintWriter(fw);
	pw.println("digraph UniverseRuntime {");
		
	Runtime.getRuntime().addShutdownHook(new Thread() {
		public void run() {
		    cleanup();
		}
	    });
    }
	
    /**
     * Generates a name for an object.
     * 
     * @param o	the object
     * @return	the name
     */
    public String getName (Object o) {
	String hashCode = Integer.toString(System.identityHashCode(o));
	String clazz = o.getClass().getName();
	int index = clazz.lastIndexOf(".");

	if ( index == -1 )
	    return clazz + "@" + hashCode;
	else
	    return clazz.substring(index + 1) + "@" + hashCode;
    }
	
    public void setOwnerRep (Object obj, Object owner) {
	/*
	 * If pw is initialized, everything is fine, if not, we have to save the
	 * data elsewhere until the constructor is called.
	 */
	if ( pw != null ) {
	    String ownerName = getName(owner);
	    String objName = getName(obj);

	    /*
	     * Don't display the head of the structure (the "virtual" owner of
	     * the root set).
	     */
	    if ( !(owner instanceof UrtHashtable) )
		pw.println("\t\"" + ownerName + "\" -> \"" + objName + "\";");

	    pw.println("\t\"" + objName + "\"[label=\"" + (counter++) + " - " + objName + "\"];");
	    pw.println("\tsubgraph \"cluster" + ownerName + "\" {\"" + objName + "\";}");
	    pw.flush();
	} else {
	    /*
	     * The objects registered in the superclass constructor don't need
	     * to be displayed. It's just the Thread and the Queue both of which
	     * should never have objects in their universe.
	     * If they have it's a bug. :)
	     * And it will be shown nevertheless (when the objects that reference
	     * the thread or the queue as owner are added).
	     */
	}
	super.setOwnerRep(obj, owner);
    }
	
    /**
     * Finish the graph and close the file.
     */
    private void cleanup () {
	pw.println("}");
	pw.flush();
	try {
	    fw.close();
	} catch ( IOException e ) {}
    }
}
