/*
 * This file is part the Universe Runtime Classes.
 *
 * Copyright (C) 2003-2005 Swiss Federal Institute of Technology Zurich
 *
 * Part of mjc, the MultiJava Compiler.
 *
 * Copyright (C) 2000-2005 Iowa State University
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: UrtDefaultImplementation.java,v 1.2 2005/06/28 20:56:08 wdietl Exp $
 */

package org.multijava.universes.rt.impl;

import java.lang.ref.ReferenceQueue;

import org.multijava.mjc.MjcTokenTypes;
import org.multijava.universes.rt.UniverseRuntime;
import org.multijava.universes.rt.UrtImplementation;
import org.multijava.util.Utils;

/**
 * This Class manages the dynamic type checking for the universe type
 * system. It uses an optimized IntHashtable (UrtHashtable)
 * to store the ownership relations.
 * 
 * Newly created objects should be registered to the hashtable by either
 * calling setConstructorData before the object is created and then calling
 * setOwner in the constructor (before field initializations!!) or by
 * calling setOwnerPeer or setOwnerRep after the object has been created.
 * 
 * Whenever there's a universe aware constructor, the first method has to
 * be used in order to make sure objects created in the constructor and
 * field initializations can be registered too.
 * 
 * Furthermore it provides functions to check if two objects are peers or
 * if one is the owner of the other.
 * Functions to handle errors are implemented too, so alternative
 * implementations of this class can handle errors differently.
 * 
 * Alternative implementations have to implement
 * UrtImplementation.
 * 
 * @author scdaniel
 */
public class UrtDefaultImplementation extends Utils implements UrtImplementation {
    // the hashtable used to store the ownership relations
    private UrtHashtable table;

    // the ReferenceQueue that "informs" us about deleted
    // objects so we can cleanup
    private ReferenceQueue queue;
	
    /**
     * Constructor.
     * Initializes things and sets up the "root" of the ownership relation
     * hierarchy by inserting a "root" element and the current thread into
     * the hashtable.
     * We only need this to be able to handle objects in the root set the
     * same way as normal objects.
     * (Objects in the root set don't have an owner; Examples: the first
     * Thread that is created and runs the program or peer objects created
     * in the static (!) main function.)
     */
    public UrtDefaultImplementation () {
	table = new UrtHashtable();
	queue = new ReferenceQueue();
		
	/* put the table itself on top of the hierarchy */
	UrtHashtableEntry e = table.put(table, new UrtHashtableEntry());
	e.obj = new UrtWeakReference(table);
	// Make a circular reference in order to cope with Scala - object's
	// are created by 'legacy' code, but not static in the sense of Java.
	e.owner = e;
		
	/* add the current thread as the first object to the root-set */
	setOwnerRep(Thread.currentThread(), table);
	setOwnerRep(queue, table);
    }

    /**
     * Registers the ownership relation of obj and owner in the hashtable.
     * (owner is the owner of obj)
     * 
     * @param obj	the object
     * @param owner	it's owner
     */
    public void setOwnerRep (Object obj, Object owner) {
	assertTrue(obj != null);
	assertTrue(owner != null);
		
	/* try to put the object into the hashtable */
	UrtHashtableEntry e;
	if ( obj instanceof Thread )
	    e = setOwner(obj, owner, new UrtHashtableThreadEntry(obj, queue));
	else
	    e = setOwner(obj, owner, new UrtHashtableEntry());
    }

    /**
     * Registers the ownership relation of obj and current in the hashtable.
     * (they both share the same owner)
     * 
     * @param obj		the object
     * @param current	the owner of the current universe when obj is created
     */
    public void setOwnerPeer (Object obj, Object current) {
	assertTrue(obj != null);
	assertTrue(current != null);
	
	/* get owner of the current object ... */
	UrtHashtableEntry e = table.get(current);
	assertTrue(e != null);
	assertTrue(e.owner != null,"No owner exists");
	// FIXME: potential problem if owner of current object
	// already collected
	setOwnerRep(obj, e.owner.obj.get());
    }
	
    /**
     * Saves the current object and the modifier for the object being
     * created in its context in the current threads hashtable entry.
     * This information is later retrieved by setOwner (which is called in
     * the constructor) to set the owner of newly created objects before
     * they create any other objects in the constructor.
     * 
     * @param currentObject	the object that owns the current universe
     * @param objectClass	the class of the object that is to be created
     * @param modifier		the modifier that applies to the object to be created
     * 				(one of MjcTokenTypes.LITERAL_peer, MjcTokenTypes.LITERAL_rep)
     */
    public void setConstructorData (Object currentObject, Object objectClass, int modifier) {
	assertTrue(currentObject != null);
	assertTrue(objectClass != null);

	UrtHashtableThreadEntry e = getCurrentThreadEntry();
		
	e.currentObject = currentObject;
	e.objectClass = objectClass;
	e.modifier = modifier;
    }
	
    /**
     * Sets the owner of the current object by using the data stored in the
     * the current threads hashtable entry.
     * This function should be called in the constructor, before any object
     * (including field initializers) is created.
     * 
     * @param o	the current object
     */
    public void setOwner (Object o) {
	assertTrue(o != null);

	UrtHashtableThreadEntry e = getCurrentThreadEntry();
		
	/* there has to be some data associated with this threads entry */
	assertTrue(e.currentObject != null);
	assertTrue(e.objectClass != null);
		
	/* 
	 * If the classes do not match, the data has been saved in the threads
	 * entry before the creation of a native Java Object (not universe aware)
	 * and the Object being created now (universe aware) is created from
	 * outside of universe aware code. Therefore the saved date does not
	 * apply.
	 * 
	 * Get the owner for this Object and if no special owner is defined,
	 * register it as peer relative to the last object.
	 */
	if ( e.objectClass != o.getClass() ) {
	    Object owner = UniverseRuntime.policy.getNativeOwner(o);
	    if ( owner == null )
		setOwnerPeer(o, e.currentObject);
	    else
		setOwnerRep(o, owner);
			
	    return;
	}
		
	/* call the right function */
	switch ( e.modifier ) {
	case MjcTokenTypes.LITERAL_peer:
	    setOwnerPeer(o, e.currentObject); break;
	case MjcTokenTypes.LITERAL_rep:
	    setOwnerRep(o, e.currentObject); break;
	default:
	    assertTrue(e.modifier != e.modifier); break;
	}
    }

    /**
     * Registers the ownership relation of the array obj and the object
     * owner in the hashtable. (owner is the owner of obj)
     * And stores the universe type of the array elements in the hashtable
     * entry that is used for the object.
     * 
     * @param obj	the object
     * @param owner	it's owner
     */
    public void setArrayOwnerRep (Object obj, Object owner, int arrayElementType) {
	assertTrue(obj != null);
	assertTrue(owner != null);
		
	UrtHashtableArrayEntry e = (UrtHashtableArrayEntry)
	    setOwner(obj, owner, new UrtHashtableArrayEntry(arrayElementType));
    }

    /**
     * Registers the ownership relation of the array obj and the object
     * current in the hashtable. (they both share the same owner)
     * And stores the universe type of the array elements in the hashtable
     * entry that is used for the object.
     * 
     * @param obj		the array
     * @param current	the owner of the current universe when obj is created
     */
    public void setArrayOwnerPeer (Object obj, Object current, int arrayElementType) {
	assertTrue(obj != null);
	assertTrue(current != null);
		
	/* get owner of the current object ... */
	UrtHashtableEntry e = table.get(current);
	assertTrue(e != null);

	// FIXME: potential problem if owner of current object
	// already collected
	setArrayOwnerRep(obj, e.owner.obj.get(), arrayElementType);
    }
	
    /**
     * Checks if the array specified has exactly the elementType specified.
     * 
     * For ArrayStores, we need to know whether it is really readonly.
     * That's why it does not work the same as instanceof and therefor
     * returns only true if the actual element type is equal to elementType. 
     * 
     * @param o				the array to check
     * @param elementType	the element type to check for
     * @return	true if the elementType of o equals elementType
     */
    public boolean checkArrayType (Object o, int elementType) {
	/* null objects don't have elementTypes */
	if ( o == null )
	    return false;

	UrtHashtableArrayEntry e =
	    (UrtHashtableArrayEntry) table.get(o);
		
	/*
	 * if the object is not in the hashtable, handle external objects
	 * as defined in isExternalPeer() if elementType is peer.
	 * Otherwise if element type is rep return false and if it is
	 * readonly return true.
	 */
	if ( e == null )
	    if ( elementType == MjcTokenTypes.LITERAL_peer )
		return UniverseRuntime.policy.isExternalPeer();
	    else if ( elementType == MjcTokenTypes.LITERAL_rep )
		return false;
	    else
		return true;
		
	return (e.elementType == elementType);
    }
	
    /**
     * This function test whether the two objects specified are peers.
     * 
     * @param o1
     * @param o2
     * @return	true if o1 and o2 share the same owner
     */
    public boolean isPeer (Object o1, Object o2) {
	/* null objects don't have the same parent as anything else */
	if ( (o1 == null) || (o2 == null) )
	    return false;
		
	UrtHashtableEntry e1 = getOwnerEntry(o1);
	UrtHashtableEntry e2 = getOwnerEntry(o2);

	/*
	 *	was one of the objects created outside of universe aware code?
	 */
	boolean result;
	if ( (e1 == null) || (e2 == null) )
	    return UniverseRuntime.policy.isExternalPeer();
		
	return (e1 == e2);
    }
	
    /**
     * Tests whether owner is the owner of obj.
     * 
     * @param owner
     * @param child
     * @return	true if owner is the owner of obj
     */
    public boolean isOwner (Object owner, Object obj) {
	/* null objects don't have or are owners */
	if ( (owner == null) || (obj == null) )
	    return false;
		
	UrtHashtableEntry r1 = table.get(owner);
	UrtHashtableEntry r2 = getOwnerEntry(obj);

	/*
	 * if the object pretending to be child was created outside
	 * of universe aware code, so owner can't be it's owner.
	 * The same holds if the owner was not found.
	 */
	if ( (r1 == null) || (r2 == null) )
	    return false;
		
	return (r1 == r2);
    }
	
    /**
     * Returns the owner of an object.
     * 
     * Might be null if the owner is not in registered to the hashtable. 
     * 
     * @param obj	the object
     * @return		the owner for obj
     */
    public Object getOwner (Object obj) {
	assertTrue(obj != null);
		
	UrtHashtableEntry ownerEntry = getOwnerEntry(obj);
		
	if ( ownerEntry == null )
	    return null;
		
	/* what to do if the owner field is null? */
	if ( ownerEntry.obj == null )
	    return UniverseRuntime.policy.getCollectedOwner(obj);
		
	return ownerEntry.obj.get();
    }

    /**
     * Get a member of the rootset. To be able to put objects to the rootset
     * (for static initializers) or do things relative to the rootset (for
     * example in the policy).
     * 
     * @return a member of the root set
     */
    public Object getRootSetMember () {
	return queue;
    }

    /**
     * Saves the current context in the current threads hashtable entry.
     * This information is later retrieved by getContext to run static
     * functions in the same universe as the function they were called from.
     * (static functions use getContext() instead of the not existing "this"
     * reference)
     * 
     * @param currentObject	the current context
     */
    public void setContext (Object currentObject) {
	assertTrue(currentObject != null);

	UrtHashtableThreadEntry e = getCurrentThreadEntry();
		
	e.push(currentObject);
    }

    /**
     * Resets the saved context to the value it had before the
     * last setContext(). i.e. pops the top of the stack.
     */
    public void resetContext () {
	UrtHashtableThreadEntry e = getCurrentThreadEntry();
		
	e.pop();
    }

    /**
     * Returns the current context.
     * 
     * @return the current context
     */
    public Object getContext () {
	UrtHashtableThreadEntry e = getCurrentThreadEntry();
		
	return e.getCurrentContext();
    }
	
	
    /* ---------- PRIVATE FUNCTIONS ---------- */

    /**
     * Internal function that does the actual insert into the hashtable.
     * Avoids code duplication.
     * 
     * To allow different object types to have more data associated with
     * them, the hashtable entry that should be used for the object has
     * to be given. (ie. Arrays store the elementType, ...)
     * 
     * @param obj		the object to insert
     * @param owner		the owner of that object
     * @param e			the hashtable entry to use for that object
     * 
     * @return	the hashtable entry of the object
     */
    private UrtHashtableEntry setOwner (Object obj, Object owner,
					UrtHashtableEntry e) {
	/* try to put the object into the hashtable */
	e = table.put(obj, e);
		
	/* if it was already registered, exit */
	if ( (e.owner != null) )
	    return e;
		
	/* 
	 *	Create a weak reference to the object and register
	 *	with the reference queue. To be noticed, when the
	 *	object is collected.
	 */
	e.obj = new UrtWeakReference(obj, queue);
		
	/* Set owner */
	e.owner = table.get(owner);
	/* we need to have an owner here */
	assertTrue(e.owner != null);
		
	e.owner.children++;
		
	/* 
	 * Reset the threads saved data, so constructors or functions called
	 * from non-universe aware code are handled correctly.
	 */
	UrtHashtableThreadEntry thread = getCurrentThreadEntry();
	thread.currentObject = owner;
	thread.objectClass = Object.class;
	thread.modifier = MjcTokenTypes.LITERAL_peer;
		
	/*
	 *	Check for collected objects and remove unnecessary
	 *	stuff in our data structure to make it ready to be
	 *	collected too.
	 */
	UrtWeakReference o;
	UrtHashtableEntry i;
	while ( (o = (UrtWeakReference) queue.poll()) != null ) {
	    i = (UrtHashtableEntry) table.get(o);
	    /* clear WeakLink so the Object can be collected */
	    i.obj = null;
	    /* the second test is for further iterations */
	    while ( (i.children == 0) && (i.obj == null) ) {
		table.remove(i);
		i.owner.children--;
				
		/*
		 * If the owner has no more children now, and
		 * its object already has been collected, we
		 * may delete this one too.
		 */
		i = i.owner;
				
		if ( i == null )
		    break;
	    }
	}
		
	return e;
    }
	
    /**
     * Internal function that gets the hashtable entry of an objects owner.
     * 
     * @param obj	the object
     * @return		the hashtable entry of that objects owner
     */
    private UrtHashtableEntry getOwnerEntry (Object obj) {
	UrtHashtableEntry e = table.get(obj);
		
	/* 
	 * every object should be in our table
	 * if not, it has been created outside of universe aware code
	 */
	if ( e == null )
	    return null;
		
	/* every object should have an owner */
	assertTrue(e.owner != null);
		
	return e.owner;
    }
	
    /**
     * Get the current Threads HashtableEntry and create it
     * if it does not exist.
     * (It may not exist if the Thread has been created outside of universe
     * aware code)
     * 
     * @return the current Threads UrtHashtableThreadEntry
     */
    private UrtHashtableThreadEntry getCurrentThreadEntry () {
	UrtHashtableThreadEntry thread =
	    (UrtHashtableThreadEntry) table.get(Thread.currentThread());
	if ( thread == null ) {
	    setOwnerRep(Thread.currentThread(), table);
	    thread =
		(UrtHashtableThreadEntry) table.get(Thread.currentThread());
	}
		
	return thread;
    }
}
