/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * 
 * $Id: ExecutionTestFactory.scala 817 2008-01-25 19:44:05Z ms $
 */
package ch.ethz.inf.sct.uts.test.helpers

import scala.tools.nsc.Main
import java.io.File
import java.lang.{ClassNotFoundException, NoSuchMethodException}
import java.lang.reflect.InvocationTargetException
import java.net.URL
import scala.tools.nsc.util.ClassPath
import scala.tools.nsc.{GenericRunnerCommand,ObjectRunner}
import scala.testing.SUnit
import SUnit._

/**
 * Produce test cases which execute a given class' main method.
 * 
 * @author  Manfred Stock
 * @version $Revision: 817 $
 */
trait ExecutionTestFactory extends TestUtils {
  /**
   * Classpath to pass to the runner.
   */
  val classPath : List[String]
  
  /**
   * Create an instance of the testcase using given testdata.
   * @param cls The class which should be executed.
   * @param n Some name for the test.
   * @param exception If an exception is expected.
   * @return A testcase.
   */
  def produce(cls: String, n: String, exception: Boolean) : ExecutionTest = {
    new ExecutionTest(cls, n, exception)
  }
  
  /**
   * Class implementing a <code>TestCase</code> which tries to execute the 
   * main method of a given class. Fails if any exceptions are thrown.
   * @param cls Name of the class including package. 
   * @param n Name for this test case.
   * @param exception If an exception is expected.
   */
  class ExecutionTest(cls: String, n: String, exception: Boolean) extends TestCase(n) {
    override def runTest() = {
      val info = "Executing "+cls+"..."
      val sharps = line('#',info,120)
      println(sharps)
      println(info)
      println(sharps)
      
      // Setup classpath for the test execution.
      val cp: List[URL] =
        for (
          file <- ClassPath.expandPath(classPath mkString ("",":","")) map (new File(_)) if file.exists;
          val url = try {
              Some(file.toURL) 
            }
            catch { 
              case e => Console.println(e); None 
            }
          if !url.isEmpty
        ) yield url.get

      try {
        // Actually execute the given class
        ObjectRunner.run(cp, cls, List())  
      }
      catch {
        case e: InvocationTargetException => 
          Console.err.println(e.getCause)
          e.getCause match {
            case e: ExceptionInInitializerError =>
              e.printStackTrace
            case _                              =>
          }
          if (!exception) fail("Execution of "+cls+" failed: "+e.getCause)
        case e                            =>
          Console.err.println(e)
          if (!exception) fail("Execution of "+cls+" failed: "+e)
      } 
    }
  }
}
