/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * 
 * $Id: RuntimeComponent.scala 883 2008-02-01 18:59:56Z ms $
 */
package ch.ethz.inf.sct.uts.plugin.runtimecheck

import scala.tools.nsc._
import scala.tools.nsc.transform._
import scala.tools.nsc.plugins.PluginComponent
import ch.ethz.inf.sct.uts.plugin._
import ch.ethz.inf.sct.uts.plugin.common._
import ch.ethz.inf.sct.uts.annotation._

/**
 * Main part of the Universe type system runtime checker. It traverses the tree and 
 * inserts the runtime checks.
 * 
 * @author  Manfred Stock
 * @version $Revision: 883 $
 */
class RuntimeComponent (glbl: Global) extends UniverseTypeSystemComponentBase(glbl) {
  import global._
  import global.definitions._

  /**
   * Name of this compiler phase.
   */
  val phaseName = "uts-runtime"
  
  /**
   * When to execute this phase. Currently done after running the static checks.
   */
  val runsAfter = "uts-static"
  
  /**
   * Factory to create the new phase for the Universe Type System Runtime checks.
   */
  def newPhase(prev: Phase) = new RuntimeCheckAdditionPhase(prev)
 
  /** 
   * Flag if the runtime Library implemented in Scala should be 
   * used. Defaults to 'yes'.
   */
  private var scruntime = true
  
  /**
   * Use the MultiJava runtime check support library.
   */
  def setMJRuntime {
     scruntime = false
  }
  
  /**
   * Use the Scala runtime check support library.
   */
  def setSCRuntime {
    scruntime = true
  }
  
  /**
   * Phase to add runtime checks for the Universe Type System to the AST.
   * @param prev The previous phase.
   */
  class RuntimeCheckAdditionPhase(prev: Phase) extends Phase(prev) {
    /**
     * Name of the phase.
     */
    val name = phaseName
    
    /**
     * Object with a <code>Transform</code> instance, used to create new transformers.
     */
    val transform = if (scruntime) {
      new RuntimeCheckTransform {
        val global : RuntimeComponent.this.global.type = RuntimeComponent.this.global
        val logger : UTSLogger                         = RuntimeComponent.this.logger 
      }
    }
    else {
      new MJRuntimeCheckTransform {
        val global : RuntimeComponent.this.global.type = RuntimeComponent.this.global
        val logger : UTSLogger                         = RuntimeComponent.this.logger 
      }
    }
    
    /**
     * Add runtime checks to a given unit.
     * @param unit The unit where the checks should be added.
     */
    def processUnit(unit: CompilationUnit) {
      val transformer = transform.newTransformer(unit)
      unit.body = transformer.transform(unit.body)
    }
      
    /**
     * Run the Universe Type System checking phase.
     */
    def run {
      processUnits(
          "Adding Universe Type System runtime checks...",
          currentRun.units foreach processUnit
      )
    }
  }
}