/*
 * This file is part the Universe Runtime Classes.
 *
 * Copyright (C) 2003-2005 Swiss Federal Institute of Technology Zurich
 *
 * Part of mjc, the MultiJava Compiler.
 *
 * Copyright (C) 2000-2005 Iowa State University
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: UrtImplementation.java,v 1.5 2007/06/27 22:44:39 chalin Exp $
 */

package org.multijava.universes.rt;

/**
 * This interface has to be implemented by any class that is meant to be
 * used as implementation for the Universe Runtime Checks.
 * 
 * The default implementation (UrtDefaultImplementation) uses an optimized
 * hashtable to manage the objects and their relations.
 * 
 * @author scdaniel
 */
public interface UrtImplementation {
    /**
     * Registers the ownership relation of obj and owner in the hashtable.
     * (owner is the owner of obj)
     * 
     * @param obj	the object
     * @param owner	it's owner
     */
    void setOwnerRep( /*@ non_null @*/ Object obj, 
		      /*@ non_null @*/ Object owner );

    /**
     * Registers the ownership relation of obj and current in the hashtable.
     * (they both share the same owner)
     * 
     * @param obj	the object
     * @param current	the owner of the current universe when obj is created
     */
    void setOwnerPeer( /*@ non_null @*/ Object obj, /*@ non_null @*/ Object current );

    /**
     * Saves the current object and the modifier for the object
     * being created in its context in the current threads
     * hashtable entry.
     * This information is later retrieved by setOwner (which is called in the
     * constructor) to set the owner of newly created objects before they
     * create any other objects in the constructor.
     * 
     * @param currentObject	the object that owns the current universe
     * @param objectClass	the class of the object that is to be created
     * @param modifier		the modifier that applies to the object 
     * 				to be created (one of MjcTokenTypes.LITERAL_peer, MjcTokenTypes.LITERAL_rep)
     */
    void setConstructorData( /*@ non_null @*/ Object currentObject, 
			     /*@ non_null @*/ Object objectClass, 
			     int modifier );

    /**
     * Sets the owner of the current object by using the data stored in the
     * the current threads hashtable entry.
     * This function should be called in the constructor, before any object
     * (including field initializers) is created.
     * 
     * @param o	the current object
     */
    void setOwner( /*@ non_null @*/ Object o );

    /**
     * Registers the ownership relation of the array obj and the object owner
     * in the hashtable. (owner is the owner of obj)
     * And stores the universe type of the array elements in the hashtable entry
     * that is used for the object.
     * 
     * @param obj	the object
     * @param owner	it's owner
     */
    void setArrayOwnerRep( /*@ non_null @*/ Object obj, 
			   /*@ non_null @*/ Object owner,
			   int arrayElementType );

    /**
     * Registers the ownership relation of the array obj and the object current
     * in the hashtable. (they both share the same owner)
     * And stores the universe type of the array elements in the hashtable
     * entry that is used for the object.
     * 
     * @param obj		the array
     * @param current	the owner of the current universe when obj is created
     */
    void setArrayOwnerPeer( /*@ non_null @*/ Object obj,
			    /*@ non_null @*/ Object current,
			    int arrayElementType );

    /**
     * Checks if the array specified has *exactly* the elementType specified.
     * 
     * For ArrayStores, we need to know whether it is really readonly. That's
     * why it does not work the same as instanceof and therefore returns only
     * true if the actual element type is equal to elementType. 
     * 
     * @param o			the array to check
     * @param elementType	the element type to check for
     * @return	true if the elementType of o equals elementType
     */
    boolean checkArrayType( /*@ nullable @*/ Object o, int elementType );

    /**
     * This function test whether the two objects specified are peers.
     * 
     * @param o1
     * @param o2
     * @return	true if o1 and o2 share the same owner
     */
    boolean isPeer( /*@ nullable @*/ Object o1, 
		    /*@ nullable @*/ Object o2 );

    /**
     * Tests whether owner is the owner of obj.
     * 
     * @param owner
     * @param obj
     * @return	true if owner is the owner of obj
     */
    boolean isOwner( /*@ nullable @*/ Object owner, /*@ nullable @*/ Object obj );
	
    /**
     * Returns the owner of an object.
     * 
     * Might be null if the owner is not in registered to the hashtable. 
     * 
     * @param obj	the object
     * @return		the owner for obj
     */
    /*@ nullable @*/ Object getOwner( /*@ non_null @*/ Object obj );
	
    /**
     * Get a member of the rootset. To be able to put objects to the rootset
     * (for static initializers) or do things relative to the rootset (for
     * example in the policy).
     * 
     * @return a member of the root set
     */
    /*@ nullable @*/ Object getRootSetMember();
	
    /**
     * Saves the current context in the current threads hashtable entry.
     * This information is later retrieved by getContext to run static
     * functions in the same universe as the function they were called from.
     * (static functions use getContext() instead of the not existing "this"
     * reference)
     * 
     * @param currentObject	the current context
     */
    void setContext( /*@ non_null @*/ Object currentObject );
	
    /**
     * Resets the saved context to the value it had before the last
     * setContext(). i.e. pops the top of the stack.
     */
    void resetContext();
	
    /**
     * Returns the current context.
     * 
     * @return the current context
     */
    /*@ nullable @*/ Object getContext();
}
