/*
 * This file is part the Universe Runtime Classes.
 *
 * Copyright (C) 2003-2005 Swiss Federal Institute of Technology Zurich
 *
 * Part of mjc, the MultiJava Compiler.
 *
 * Copyright (C) 2000-2005 Iowa State University
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: UniverseRuntime.java,v 1.3 2007/06/27 14:26:38 chalin Exp $
 */

package org.multijava.universes.rt;

import org.multijava.universes.rt.impl.*;
import org.multijava.universes.rt.policy.*;

/**
 * This Class manages the dynamic type checking for the universe type
 * system.
 * 
 * This class just defines to static fields and a static initializer where
 * the real runtime classes are instantiated.
 * 
 * There are two types of runtime classes: the implementation class that takes
 * care of the objects (the default implementation (UrtDefaultImplementation)
 * uses an optimized hashtable) and the policy class (default: UrtDefaultPolicy)
 * that defines what happens when dynamic checks fail etc.
 * 
 * To create your own implementations of one of these you just need your class
 * to implement the respective Interface:
 *  - UrtImplementation
 *  - UrtPolicy
 * 
 * @author scdaniel
 */
public class UniverseRuntime {
    /**
     * this field stores at runtime a reference to an instance of a
     * class to be used as handler (hashtable) for the RuntimeChecks.
     */  
    public static /*@non_null*/ UrtImplementation handler;

    /**
     * this field stores at runtime a reference to an instance of a
     * class to be used as policy for the RuntimeChecks.
     */  
    public static /*@non_null*/ UrtPolicy policy;
	
    /**
     * Static initializer.
     * Instantiates an object of the class specified by the JVM
     * command-line switch -DUrtImplementation=<class> and stores it into
     * the static field handler and instiantiates an object of the class
     * specified by -DurtPolicy=<class> and stores it into the static field
     * policy.
     * Defaults to impl.UrtDefaultImplementation and policy.UrtDefaultPolicy
     * if nothing is specified or instantiation fails.
     */
    static {
	/* get name of implementation class */
	String runtimeClass = System.getProperty("UrtImplementation");
	if ( runtimeClass != null ) {
	    /* try to create an instance of this class */
	    try {
		handler = (UrtImplementation)
		    Class.forName(runtimeClass).newInstance();
	    } catch ( Exception e ) {
		System.err.println("WARNING: the specified Universe Runtime Implementation Class (" +
				   runtimeClass +
				   ") could not be instantiated, using the default instead.");
		// System.err.println(e);
		handler = new UrtDefaultImplementation();
	    }
	} else {
	    handler = new UrtDefaultImplementation();
	}

	/* get name of policy class */
	runtimeClass = System.getProperty("UrtPolicy");
	if ( runtimeClass != null ) {
	    /* try to create an instance of this class */
	    try {
		policy = (UrtPolicy)
		    Class.forName(runtimeClass).newInstance();
	    } catch ( Exception e ) {
		System.err.println("WARNING: the specified Universe Runtime Policy Class (" +
				   runtimeClass +
				   ") could not be instantiated, using the default instead.");
		// System.err.println(e);
		policy = new UrtDefaultPolicy();
	    }
	} else {
	    policy = new UrtDefaultPolicy();
	}
    }
}
