/*****************************************************
 * Simple program to construct aodv-packets which are 
 * then sent to the network.
 *****************************************************/

#include <sys/socket.h>
#include <netinet/in.h>
#include <stdio.h>
#include <sys/types.h>
#include <netdb.h>
#include <stdlib.h>
#include <strings.h>
#include <unistd.h>
#include <time.h>

#define _GNU_SOURCE
#include <getopt.h>

#define RREQSIZE 24
#define RREPSIZE 20

#define RREQ 1
#define RREP 2

#define DSDSTART 129
#define DSDFOLLOWING 130

#define AODVPORT 654

char* aodvtypes[] = {"","RREQ","RREP"};

void datatonet(unsigned int * data, int len){
  // Convert host byte order to network byte order
  int i;
  for(i = 0; i < len; i++){
    data[i] = htonl(data[i]);
  }
}

void help(char* prog){
  printf("Usage: %s [options] <receiver>\n"
	 "Possible options are:\n"
	 "\t--next <number of extensions to send>\n"
	 "\t--rreq\n"
	 "\t--rrep\n"
	 "\t--port <port to send packet to, default %i>\n"
	 "\t--type <type to give to extensions>\n"
	 "\t--help\n",prog,AODVPORT);
}

int main(int argc, char** argv){
  int sock,size,c,i,offset;
  struct sockaddr_in server, localhost;
  struct hostent *hp;

  unsigned int* data;

  // parse command line options
  char* receiver;
  int nExt = 0;
  int pktsize = RREQSIZE;
  unsigned char exttype = DSDSTART;
  unsigned char type = RREQ;
  int port = AODVPORT;
  while (1){
    int option_index = 0;
    static struct option long_options[] = {
      {"next",1,NULL,'n'},
      {"rreq",0,NULL,RREQ},
      {"rrep",0,NULL,RREP},
      {"type",1,NULL,'t'},
      {"port",1,NULL,'P'},
      {"help",0,NULL,'h'},
      {0,0,0,0}
    };
    c = getopt_long (argc, argv, "n:qpt:P:h", long_options, &option_index);
    if (c == -1)
      break;

    switch(c){
    case 'n':
      nExt = atoi(optarg);
      break;
    case RREQ:
    case 'q':
      type = RREQ;
      pktsize = RREQSIZE;
      break;
    case RREP:
    case 'p':
      type = RREP;
      pktsize = RREPSIZE;
      break;
    case 't':
      exttype = (unsigned char) atoi(optarg);
      break;
    case 'P':
      port = atoi(optarg);
      break;
    case 'h':
      help(argv[0]);
      exit(1);
      break;
    }
  }
  // last argument is receiver
  if(optind == argc){
    help(argv[0]);
    exit(1);
  }else{
    receiver = argv[optind];
  }

  // set up connection
  if((sock = socket(AF_INET,SOCK_DGRAM,0)) == -1){
    perror("socket");
    exit(1);
  }

  server.sin_family = AF_INET;
  if((hp = gethostbyname(receiver)) == NULL){
    herror("gethostbyname");
    exit(1);
  }

  /* copy internet address to server address */
  bcopy(hp->h_addr, &server.sin_addr, hp->h_length);
  /* set port */
  server.sin_port = htons(port);

  if((connect(sock,(struct sockaddr*) &server,sizeof(struct sockaddr_in))) < 0){
    perror("connect");
    exit(1);
  }

  // we send nExt extensions which are one word in size (type, length, 2 byte data)
  pktsize += (2+2)*nExt;
  printf("Sending a %i byte %s-packet, %i extensions of type %i, to port %i... ",pktsize,aodvtypes[type],nExt,exttype,port);
  data = (unsigned int*) malloc(sizeof(unsigned char)*pktsize);
  // Construct packet data...
  // type
  data[0] = type << 24;
  // rest of first word stays 0 for now...
  size = sizeof(localhost);
  getsockname(sock,(struct sockaddr*) &localhost,&size);
  // now insert different data according to type
  if(type == RREQ){
    // RREQ_ID - set to current time
    data[1] = time(0);
    // Destination IP address - 127.0.0.1
    data[2] = 0x7f000001;
    // Destination sequence number - also the current time
    data[3] = time(0);
    // Originator IP address
    data[4] = ntohl(localhost.sin_addr.s_addr);
    // Originator sequence number - again the current time
    data[5] = time(0);
    offset = 6;
  }else if(type == RREP){
    // Destination IP address - 127.0.0.1
    data[1] = 0x7f000001;
    // Destination sequence number - also the current time
    data[2] = time(0);
    // Originator IP address
    data[3] = ntohl(localhost.sin_addr.s_addr);
    // Lifetime
    data[4] = 1 << 16;
    offset = 5;
  }

  // Fill extension
  for(i = 0; i < nExt; i++){
    data[offset+i] = exttype << 24 | 2 << 16;
  }

  datatonet(data,pktsize/4);
  // finally send data
  if(write(sock,data,pktsize) == pktsize){
    printf("Done.\n");
  }else{
    printf("Failed.\n");
  }
  
  return 0;
}
