/* Shared library add-on to iptables to add aodvext match support. */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <getopt.h>

#include <iptables.h>
#include <linux/netfilter_ipv4/ip_tables.h>
#include <ipt_aodvext.h>

/* Function which prints out usage message. */
static void
help(void)
{
  printf(
	 "aodvext match v%s takes the following option:\n"
	 "[!] --invert                    Invert result\n"
	 "--aodvext                       Match only extensions\n"
	 "--switcher                      Be only switchable\n"
	 "--hello <n>                     Match every n'th HELLO-message if\n"
	 "                                in switcher-mode\n"
	 "                                (default 0 -> all, -1: none)\n"
	 "--alldebug                      Print info on packets\n"
	 "--matchdebug                    Print info on matched packets\n"
	 "  => Only one of --aodvext or --switcher can be used!\n",
	 IPTABLES_VERSION);
}

static struct option opts[] = {
  { "invert", 0, 0, 'i' },
  { "aodvext", 0, 0, 'a' },
  { "switcher", 0, 0, 's' },
  { "hello", 1, 0, 'H' },
  { "matchdebug", 0, 0, 'm' },
  { "alldebug", 0, 0, 'd' },
  { 0 }
};

/* Initialize the target. */
static void
init(struct ipt_entry_match *t, unsigned int *nfcache)
{
  struct ipt_aodvext_info *info = (struct ipt_aodvext_info *)t->data;
  info->switcher = 0;
  info->aodvext = 1;
  info->hello = 0;
  info->invert = 0;
  info->matchdebug = 0;
  info->alldebug = 0;
}

/* Function which parses command options; returns true if it
   ate an option */
static int
parse(int c, char **argv, int invert, unsigned int *flags,
      const struct ipt_entry *entry,
      unsigned int *nfcache,
      struct ipt_entry_match **match)
{
  struct ipt_aodvext_info *info = (struct ipt_aodvext_info *)(*match)->data;
  switch (c){
  case 'i':
    // --invert should only be given once
    if (*flags == 1)
      exit_error(PARAMETER_PROBLEM, "aodvext match: only use --invert once!");
    info->invert = 1;
    // check if ! was given
    check_inverse(optarg, &invert, &optind, 0);
    if (invert)
      info->invert = 0;
    *flags = 1;
    break;
  case 'a':
    // turn matching of aodv extensions on
    info->switcher = 0;
    info->aodvext = 1;
    break;
  case 's':
    // turn switcher on
    info->aodvext = 0;
    info->switcher = 1;
    break;
  case 'm':
    // turn match debugging on
    info->matchdebug = 1;
    break;
  case 'd':
    // turn all debugging on
    info->alldebug = 1;
    break; 
  case 'H':
    // turn on matching of hello messages in switcher mode
    if(optarg != NULL){
      info->hello = atoi(optarg);
    }else{
      exit_error(PARAMETER_PROBLEM, "--hello needs a numeric argument!");
    }
    break; 
  default:
    return 0;
  }
  return 1;
}

static void
final_check(unsigned int flags)
{
}

static void
print(const struct ipt_ip *ip,
      const struct ipt_entry_match *match,
      int numeric){
  const struct ipt_aodvext_info *info = (const struct ipt_aodvext_info *)match->data;
  int sock,val,size;

  if (info->invert)
    printf("!");
  if (info->aodvext)
    printf("AODV extensions ");
  else if(info->switcher){
    // create socket
    if((sock = socket(AF_INET,SOCK_RAW,IPPROTO_RAW)) == -1){
      perror("socket");
      val = -1;
    }

    size = sizeof(int);
    if(getsockopt(sock, IPPROTO_IP, RS_SOCKOPTNUM, &val, &size) == -1){
      perror("getsockopt");
      val = -1;
    }
    
    printf("ruleswitcher: %i ",val);
  }
  if (info->hello >= 0 && info->switcher){
    printf("hello: %d'th ",info->hello);
  }
  if (info->matchdebug)
    printf("md ");
  if (info->alldebug)
    printf("ad ");
}

static void
save(const struct ipt_ip *ip, const struct ipt_entry_match *match)
{
  const struct ipt_aodvext_info *info = (const struct ipt_aodvext_info *)match->data;

  if (info->invert)
    printf("--invert ");
  if (info->aodvext)
    printf("--aodvext ");
  if (info->switcher)
    printf("--switcher ");
  if (info->hello >= 0)
    printf("--hello %d",info->hello);
  if (info->matchdebug)
    printf("--matchdebug ");
  if (info->alldebug)
    printf("--alldebug ");
}

static struct iptables_match aodvext = {
  .next		= NULL,
  .name		= "aodvext",
  .version	= IPTABLES_VERSION,
  .size		= IPT_ALIGN(sizeof(struct ipt_aodvext_info)),
  .userspacesize= IPT_ALIGN(sizeof(struct ipt_aodvext_info)),
  .help		= &help,
  .init         = &init,
  .parse	= &parse,
  .final_check 	= &final_check,
  .print	= print,
  .save		= save,
  .extra_opts	= opts
};

void __attribute__ ((constructor)) my_init(void)
{
  register_match(&aodvext);
}
